package com.teravation.labjack;
/**
 * This file is part of LabJackJava.  Copyright 2003-2004, Teravation.  All rights reserved. 
 * 
 * LabJackJava is free software; you can redistribute it and/or modify it under the terms of the GNU
 * General Public License as published by the Free Software Foundation; either version 2 of the License,
 * or (at your option) any later version.
 * 
 * LabJackJava is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; 
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along with LabJackJava;
 * if not, write to:
 *    Free Software Foundation, Inc.
 *    59 Temple Place, Suite 330
 *    Boston, MA  02111-1307  USA
 * 
 * In addition, as a special exception, Teravation gives permission to link the code of this program with
 * the LabJack C driver (or with modified versions of the LabJack C driver that use the same license),
 * and distribute linked combinations including the two.  You must obey the GNU General Public License in all
 * respects for all of the code used other than the LabJack C driver.  If you modify this file, you may 
 * extend this exception to your version of the file, but you are not obligated to do so.  If you do not wish to
 * do so, delete this exception statement from your version.
 * 
 */

/**
 * This abstract class implements {@link LabJack LabJack} behavior that is not specific to a particular platform, and is intended
 * to be subclassed by platform-specific implementation classes.
 *
 * <p />Copyright &copy; 2003-2004 <a href="http://www.teravation.com">Teravation</a>. All rights reserved.
 * 
 * @created (12/30/2002 2:25:49 PM)
 * @version 4.0
 * @author Chris Reigrut 
 */
public abstract class LabJackImpl implements LabJack {
	static protected LabJack[] labjacks;

	protected int productId;
	protected int serialNumber;
	protected int localId;

	protected float[] analogInputValues = new float[LabJackConstants.ANALOG_INPUT_CHANNELS];
	protected long[] analogInputGains = new long[LabJackConstants.ANALOG_DIFFERENTIAL_INPUT_CHANNELS];
	protected float[] analogOutputValues = new float[LabJackConstants.ANALOG_OUTPUT_CHANNELS];
	protected long digitalDValues = 0;
	protected long digitalDDirections = 0;
	protected long digitalIOValues = 0;
	protected long digitalIODirections = 0;
	protected long counter = 0;
	protected float tempC = 0.0f;
	protected float tempF = 0.0f;
	protected float humidity = 0.0f;
	protected float firmwareVersion = 0.0f;
	protected LabJackScan scan = null;
	
	/* (non-Javadoc)
	 * @see com.teravation.labjack.LabJack#getAI(int)
	 */
	public float getAI(int channel) throws LabJackException {
		if (channel < 0 || channel >= LabJackConstants.ANALOG_SINGLE_INPUT_CHANNELS) {
			throw new LabJackException(21);
		}
		return this.analogInputValues[channel];
	}
	
	/* (non-Javadoc)
	 * @see com.teravation.labjack.LabJack#getAO(int)
	 */
	public float getAO(int channel) throws LabJackException {
		if (channel < 0 || channel >= LabJackConstants.ANALOG_OUTPUT_CHANNELS) {
			throw new LabJackException(21);
		}
		return this.analogOutputValues[channel];
	}
	
	/* (non-Javadoc)
	 * @see com.teravation.labjack.LabJack#getCounter()
	 */
	public long getCounter() {
		return this.counter;
	}
	
	/* (non-Javadoc)
	 * @see com.teravation.labjack.LabJack#getD(int)
	 */
	public boolean getD(int channel) throws LabJackException {
		boolean value = false;
		if (channel < 0 || channel >= LabJackConstants.D_CHANNELS) {
			throw new LabJackException(21);
		}

		return this.getDigitalDValue(channel);
	}
	
	/* (non-Javadoc)
	 * @see com.teravation.labjack.LabJack#getDifferentialAI(int)
	 */
	public float getDifferentialAI(int channel) throws LabJackException {
		if (channel < 0 || channel >= LabJackConstants.ANALOG_DIFFERENTIAL_INPUT_CHANNELS) {
			throw new LabJackException(21);
		}
		return this.analogInputValues[LabJackConstants.ANALOG_SINGLE_INPUT_CHANNELS + channel];
	}
	
	/* (non-Javadoc)
	 * @see com.teravation.labjack.LabJack#getDifferentialAIGain(int)
	 */
	public int getDifferentialAIGain(int channel) throws LabJackException {
		if (channel < 0 || channel >= LabJackConstants.ANALOG_DIFFERENTIAL_INPUT_CHANNELS) {
			throw new LabJackException(21);
		}
		return (int) this.analogInputGains[channel];
	}
	/**
	 * Returns the specified digital D value from the bit array
	 * 
	 * @created (2/3/2003 5:08:49 PM)
	 * 
	 * @return boolean
	 * @param bit int
	 */
	protected boolean getDigitalDValue(int bit) {
		return (this.digitalDValues & (1 << bit)) > 0;
	}
	
	/**
	 * Returns the specified digital D value from the bit array
	 * 
	 * @created (2/3/2003 5:08:49 PM)
	 * 
	 * @return boolean
	 * @param bit int
	 */
	protected boolean getDigitalIOValue(int bit) {
		return (this.digitalIOValues & (1 << bit)) > 0;
	}
	
	/* (non-Javadoc)
	 * @see com.teravation.labjack.LabJack#getHumidity()
	 */
	public float getHumidity() {
		return humidity;
	}
	
	/* (non-Javadoc)
	 * @see com.teravation.labjack.LabJack#getIO(int)
	 */
	public boolean getIO(int channel) throws LabJackException {
		boolean value = false;
		if (channel < 0 || channel >= LabJackConstants.IO_CHANNELS) {
			throw new LabJackException(21);
		}

		return this.getDigitalIOValue(channel);
	}
	
	/* (non-Javadoc)
	 * @see com.teravation.labjack.LabJack#getLocalId()
	 */
	public int getLocalId() {
		return localId;
	}
	
	/* (non-Javadoc)
	 * @see com.teravation.labjack.LabJack#getProductId()
	 */
	public int getProductId() {
		return productId;
	}
	
	/* (non-Javadoc)
	 * @see com.teravation.labjack.LabJack#getSerialNumber()
	 */
	public int getSerialNumber() {
		return serialNumber;
	}
	
	/* (non-Javadoc)
	 * @see com.teravation.labjack.LabJack#getTempC()
	 */
	public float getTempC() {
		return tempC;
	}
	/* (non-Javadoc)
	 * @see com.teravation.labjack.LabJack#getTempF()
	 */
	public float getTempF() {
		return tempF;
	}
	
	/**
	 * Returns whether the specified D line is set for output (from the bit array)
	 * 
	 * @created (2/3/2003 5:08:49 PM)
	 * 
	 * @return boolean
	 * @param bit int
	 */
	public boolean isDForOutput(int bit) {
		return (this.digitalDDirections & (1 << bit)) > 0;
	}
	
	/**
	 * Returns whether the specified IO line is set for output (from the bit array)
	 * 
	 * @created (2/3/2003 5:08:49 PM)
	 * 
	 * @return boolean
	 * @param bit int
	 */
	public boolean isIOForOutput(int bit) {
		return (this.digitalIODirections & (1 << bit)) > 0;
	}
	
	/* (non-Javadoc)
	 * @see com.teravation.labjack.LabJack#setAO(int, float)
	 */
	public void setAO(int channel, float value) throws LabJackException {
		if (channel < 0 || channel >= LabJackConstants.ANALOG_OUTPUT_CHANNELS) {
			throw new LabJackException(21);
		}
		this.analogOutputValues[channel] = value;
	}
	
	/* (non-Javadoc)
	 * @see com.teravation.labjack.LabJack#setD(int, boolean)
	 */
	public void setD(int channel, boolean value) throws LabJackException {
		if (channel < 0 || channel >= LabJackConstants.D_CHANNELS) {
			throw new LabJackException(21);
		}
		if (!this.isDForOutput(channel)) {
			throw new LabJackException(1000);
		}
		this.setDigitalDValue(channel, value);
	}
	
	/* (non-Javadoc)
	 * @see com.teravation.labjack.LabJack#setDForInput(int)
	 */
	public void setDForInput(int channel) throws LabJackException {
		if (channel < 0 || channel >= LabJackConstants.D_CHANNELS) {
			throw new LabJackException(21);
		}
		this.digitalDDirections &= ~(1 << channel);
	}
	
	/* (non-Javadoc)
	 * @see com.teravation.labjack.LabJack#setDForOutput(int)
	 */
	public void setDForOutput(int channel) throws LabJackException {
		if (channel < 0 || channel >= LabJackConstants.D_CHANNELS) {
			throw new LabJackException(21);
		}
		this.digitalDDirections |= (1 << channel);
	}
	
	/* (non-Javadoc)
	 * @see com.teravation.labjack.LabJack#setDifferentialAIGain(int, int)
	 */
	public void setDifferentialAIGain(int channel, int gain) throws LabJackException {
		if (channel < 0 || channel >= LabJackConstants.ANALOG_DIFFERENTIAL_INPUT_CHANNELS) {
			throw new LabJackException(21);
		}
		if (gain < LabJackConstants.GAIN_1x || gain > LabJackConstants.GAIN_20x) {
			throw new LabJackException(17);
		}
		this.analogInputGains[channel] = gain;
	}
	
	/**
	 * Sets the specified digital D value in the bit array
	 * 
	 * @created (2/3/2003 5:05:40 PM)
	 * 
	 * @param bit int
	 * @param value boolean
	 */
	protected void setDigitalDValue(int bit, boolean value) {
		if (value) {
			this.digitalDValues |= (1 << bit);
		} else {
			this.digitalDValues &= ~(1 << bit);
		}
	}
	
	/**
	 * Sets the specified digital IO value in the bit array
	 * 
	 * @created (2/3/2003 5:05:40 PM)
	 * 
	 * @param bit int
	 * @param value boolean
	 */
	protected void setDigitalIOValue(int bit, boolean value) {
		if (value) {
			this.digitalIOValues |= (1 << bit);
		} else {
			this.digitalIOValues &= ~(1 << bit);
		}
	}
	
	/* (non-Javadoc)
	 * @see com.teravation.labjack.LabJack#setIO(int, boolean)
	 */
	public void setIO(int channel, boolean value) throws LabJackException {
		if (channel < 0 || channel >= LabJackConstants.IO_CHANNELS) {
			throw new LabJackException(21);
		}
		if (!this.isIOForOutput(channel)) {
			throw new LabJackException(1000);
		}
		this.setDigitalIOValue(channel, value);
	}
	
	/* (non-Javadoc)
	 * @see com.teravation.labjack.LabJack#setIOForInput(int)
	 */
	public void setIOForInput(int channel) throws LabJackException {
		if (channel < 0 || channel >= LabJackConstants.IO_CHANNELS) {
			throw new LabJackException(21);
		}
		this.digitalIODirections &= ~(1 << channel);
	}
	
	/* (non-Javadoc)
	 * @see com.teravation.labjack.LabJack#setIOForOutput(int)
	 */
	public void setIOForOutput(int channel) throws LabJackException {
		if (channel < 0 || channel >= LabJackConstants.IO_CHANNELS) {
			throw new LabJackException(21);
		}
		this.digitalIODirections |= (1 << channel);
	}
	
	/**
	 * Sets the LabJack's local id
	 * 
	 * Creation date: (1/10/2003 11:58:21 AM)
	 * @author: 
	 * 
	 * @param newLocalId int
	 */
	public void setLocalId(int newLocalId) {
		localId = newLocalId;
	}
	
	/**
	 * Sets the LabJack's product id
	 * 
	 * Creation date: (1/10/2003 11:57:21 AM)
	 * @author: 
	 * 
	 * @param newProductId int
	 */
	protected void setProductId(int newProductId) {
		productId = newProductId;
	}
	
	/**
	 * Sets the LabJack's serial number
	 * 
	 * @created (1/10/2003 11:57:59 AM)
	 * @author: 
	 * 
	 * @param newSerialNumber int
	 */
	protected void setSerialNumber(int newSerialNumber) {
		serialNumber = newSerialNumber;
	}

	/**
	 * Sets the most recent temperature (F) read by the LabJack
	 * NOTE:  Should only be used by native driver code!
	 * 
	 * @created (2/6/2003 3:35:31 PM)
	 * @author:
	 * 
	 * @param newTempF float
	 */
	protected void setTempF(float newTempF) {
		tempF = newTempF;
	}

	/**
	 * Sets the most recent temperature (C) read by the LabJack
	 * NOTE:  Should only be used by native driver code!
	 * 
	 * @created (2/6/2003 3:35:15 PM)
	 * @author:
	 *  
	 * @param newTempC float
	 */
	protected void setTempC(float newTempC) {
		tempC = newTempC;
	}

	/**
	 * Sets the current value of the counter
	 * NOTE:  Should only be used by native driver code!
	 * 
	 * @created (1/10/2003 11:58:21 AM)
	 * @author: 
	 * 
	 * @param newCounter int
	 */
	protected void setCounter(int newCounter) {
		this.counter = newCounter;
	}

	/**
	 * Sets the most recent humidity read by the LabJack
	 * NOTE:  Should only be used by native driver code!
	 * 
	 * @created (2/6/2003 3:35:48 PM)
	 * @author: 
	 * 
	 * @param newHumidity float
	 */
	protected void setHumidity(float newHumidity) {
		humidity = newHumidity;
	}

	/* (non-Javadoc)
	 * @see com.teravation.labjack.LabJack#getFirmwareVersion()
	 */
	public float getFirmwareVersion() {
		return firmwareVersion;
	}

	/**
	 * Sets the firmware version
	 * 
	 * @created (Mar 6, 2004 8:59:04 PM)
	 * @param firmwareVersion
	 */
	protected void setFirmwareVersion(float firmwareVersion) {
		this.firmwareVersion = firmwareVersion;
	}

	private String getStatusString() {
		StringBuffer sb = new StringBuffer(LabJackConstants.DIGITAL_CHANNELS + 1);
		try {
			for (int io = 0; io < LabJackConstants.IO_CHANNELS; io++) {
				if (this.isIOForOutput(io)) {
					sb.append(this.getIO(io) ? "X" : "O");
				} else {
					sb.append(this.getIO(io) ? "x" : "o");
				}
			}

			sb.append(" ");

			for (int d = 0; d < LabJackConstants.D_CHANNELS; d++) {
				if (this.isDForOutput(d)) {
					sb.append(this.getD(d) ? "X" : "O");
				} else {
					sb.append(this.getD(d) ? "x" : "o");
				}
			}
		} catch (LabJackException lje) {
		}
		return sb.toString();
	}
	
	/* (non-Javadoc)
	 * @see java.lang.Object#toString()
	 */
	public String toString() {
		return "LabJack (" + getSerialNumber() + ")";
	}

	/* (non-Javadoc)
	 * @see com.teravation.labjack.LabJack#getScan()
	 */
	public LabJackScan getScan() {
		return scan;
	}

	/* (non-Javadoc)
	 * @see com.teravation.labjack.LabJack#setPulse(float, float)
	 */
	public void setPulse(float pulseWidth, float pulseSpace) {
		this.setPulseWidth(pulseWidth);
		this.setPulseSpace(pulseSpace);
	}
}