/**
 * Name: LabJackM_Dynamic.h
 * Desc: Header file for dynamic runtime linking LJM
 * Auth: LabJack Corp.
**/

#ifndef LABJACKM_DYNAMIC_HEADER
#define LABJACKM_DYNAMIC_HEADER

/******************************************************************************
 * How To Use This Library:
 *
 * See the LJM User's Guide: labjack.com/support/ljm/users-guide
 *
 * Check out the example files for examples of workflow
 *
 * To write/read other Modbus addresses, check out labjack.com/support/modbus
 *
 *****************************************************************************/

#ifdef __cplusplus
extern "C" {
#endif

#ifdef _WIN32
	#define WIN32_LEAN_AND_MEAN
	#include <Windows.h>
#else
	#define __stdcall
	#include <dlfcn.h>
#endif

// For printf
#include <stdio.h>
#include <stdlib.h>

//For dynamic runtime linking, we first define structures that have the same
// format as the desired function prototype.
typedef int (__stdcall *tLJM_ListAll)(int DeviceType, int ConnectionType,
	int * NumFound, int * aDeviceTypes, int * aConnectionTypes,
	int * aSerialNumbers, int * aIPAddresses);

typedef int (__stdcall *tLJM_ListAllS)(const char * DeviceType, const char * ConnectionType,
	int * NumFound, int * aDeviceTypes, int * aConnectionTypes,
	int * aSerialNumbers, int * aIPAddresses);

typedef int (__stdcall *tLJM_ListAllExtended)(int DeviceType, int ConnectionType,
	int NumAddresses, const int * aAddresses, const int * aNumRegs,
	int MaxNumFound, int * NumFound, int * aDeviceTypes, int * aConnectionTypes,
	int * aSerialNumbers, int * aIPAddresses, unsigned char * aBytes);

typedef int (__stdcall *tLJM_OpenS)(const char * DeviceType, const char * ConnectionType,
	const char * Identifier, int * Handle);

typedef int (__stdcall *tLJM_Open)(int DeviceType, int ConnectionType,
	const char * Identifier, int * Handle);

typedef int (__stdcall *tLJM_GetHandleInfo)(int Handle, int * DeviceType,
	int * ConnectionType, int * SerialNumber, int * IPAddress, int * Port,
	int * MaxBytesPerMB);

typedef int (__stdcall *tLJM_Close)(int Handle);

typedef int (__stdcall *tLJM_CloseAll)(void);

typedef int (__stdcall *tLJM_CleanInfo)(int InfoHandle);

typedef int (__stdcall *tLJM_eWriteAddress)(int Handle, int Address, int Type, double Value);
typedef int (__stdcall *tLJM_eReadAddress)(int Handle, int Address, int Type, double * Value);

typedef int (__stdcall *tLJM_eWriteName)(int Handle, const char * Name, double Value);
typedef int (__stdcall *tLJM_eReadName)(int Handle, const char * Name, double * Value);

typedef int (__stdcall *tLJM_eReadAddresses)(int Handle, int NumFrames,
	const int * aAddresses, const int * aTypes, double * aValues,
	int * ErrorAddress);
typedef int (__stdcall *tLJM_eReadNames)(int Handle, int NumFrames,
	const char ** aNames, double * aValues, int * ErrorAddress);

typedef int (__stdcall *tLJM_eWriteAddresses)(int Handle, int NumFrames,
	const int * aAddresses, const int * aTypes, const double * aValues,
	int * ErrorAddress);
typedef int (__stdcall *tLJM_eWriteNames)(int Handle, int NumFrames,
	const char ** aNames, const double * aValues, int * ErrorAddress);

typedef int (__stdcall *tLJM_eReadAddressArray)(int Handle, int Address, int Type,
	int NumValues, double * aValues, int * ErrorAddress);
typedef int (__stdcall *tLJM_eReadNameArray)(int Handle, const char * Name,
	int NumValues, double * aValues, int * ErrorAddress);

typedef int (__stdcall *tLJM_eWriteAddressArray)(int Handle, int Address, int Type,
	int NumValues, const double * aValues, int * ErrorAddress);
typedef int (__stdcall *tLJM_eWriteNameArray)(int Handle, const char * Name,
	int NumValues, const double * aValues, int * ErrorAddress);

typedef int (__stdcall *tLJM_eReadAddressByteArray)(int Handle, int Address,
	int NumBytes, char * aBytes, int * ErrorAddress);
typedef int (__stdcall *tLJM_eReadNameByteArray)(int Handle, const char * Name,
	int NumBytes, char * aBytes, int * ErrorAddress);

typedef int (__stdcall *tLJM_eWriteAddressByteArray)(int Handle, int Address,
	int NumBytes, const char * aBytes, int * ErrorAddress);
typedef int (__stdcall *tLJM_eWriteNameByteArray)(int Handle, const char * Name,
	int NumBytes, const char * aBytes, int * ErrorAddress);

typedef int (__stdcall *tLJM_eAddresses)(int Handle, int NumFrames,
	const int * aAddresses, const int * aTypes, const int * aWrites,
	const int * aNumValues, double * aValues, int * ErrorAddress);
typedef int (__stdcall *tLJM_eNames)(int Handle, int NumFrames, const char ** aNames,
	const int * aWrites, const int * aNumValues, double * aValues,
	int * ErrorAddress);

typedef int (__stdcall *tLJM_eReadNameString)(int Handle, const char * Name,
	char * String);
typedef int (__stdcall *tLJM_eReadAddressString)(int Handle, int Address,
	char * String);

typedef int (__stdcall *tLJM_eWriteNameString)(int Handle, const char * Name,
	const char * String);
typedef int (__stdcall *tLJM_eWriteAddressString)(int Handle, int Address,
	const char * String);

typedef int (__stdcall *tLJM_eStreamStart)(int Handle, int ScansPerRead,
	int NumAddresses, const int * aScanList, double * ScanRate);

typedef int (__stdcall *tLJM_eStreamRead)(int Handle, double * aData,
	int * DeviceScanBacklog, int * LJMScanBacklog);

typedef int (__stdcall *tLJM_SetStreamCallback)(int Handle,
	LJM_StreamReadCallback Callback, void * Arg);

typedef int (__stdcall *tLJM_eStreamStop)(int Handle);

typedef int (__stdcall *tLJM_StreamBurst)(int Handle, int NumAddresses,
	const int * aScanList, double * ScanRate, unsigned int NumScans,
	double * aData);

typedef int (__stdcall *tLJM_GetStreamTCPReceiveBufferStatus)(int Handle,
    unsigned int * ReceiveBufferBytesSize,
    unsigned int * ReceiveBufferBytesBacklog);

typedef int (__stdcall *tLJM_InitializeAperiodicStreamOut)(int Handle,
     int StreamOutIndex,
     int TargetAddr,
     double ScanRate);

typedef int (__stdcall *tLJM_WriteAperiodicStreamOut)(int Handle,
     int StreamOutIndex,
     int NumValues,
     const double * aWriteData,
     int * LJMBufferStatus);

typedef int (__stdcall *tLJM_PeriodicStreamOut)(int Handle,
     int StreamOutIndex,
     int TargetAddr,
     double ScanRate,
     int NumValues,
     const double * aWriteData);


typedef int (__stdcall *tLJM_WriteRaw)(int Handle, const unsigned char * Data,
	int NumBytes);

typedef int (__stdcall *tLJM_ReadRaw)(int Handle, unsigned char * Data, int NumBytes);

typedef int (__stdcall *tLJM_AddressesToMBFB)(int MaxBytesPerMBFB, const int * aAddresses,
	const int * aTypes, const int * aWrites, const int * aNumValues,
	const double * aValues, int * NumFrames, unsigned char * aMBFBCommand);

typedef int (__stdcall *tLJM_MBFBComm)(int Handle, unsigned char UnitID,
	unsigned char * aMBFB, int * ErrorAddress);

typedef int (__stdcall *tLJM_UpdateValues)(unsigned char * aMBFBResponse,
	const int * aTypes, const int * aWrites, const int * aNumValues,
	int NumFrames, double * aValues);

typedef int (__stdcall *tLJM_NamesToAddresses)(int NumFrames, const char ** aNames,
	int * aAddresses, int * aTypes);

typedef int (__stdcall *tLJM_NameToAddress)(const char * Name, int * Address, int * Type);

typedef int (__stdcall *tLJM_AddressesToTypes)(int NumAddresses, int * aAddresses,
	int * aTypes);

typedef int (__stdcall *tLJM_AddressToType)(int Address, int * Type);

typedef int (__stdcall *tLJM_LookupConstantValue)(const char * Scope,
	const char * ConstantName, double * ConstantValue);

typedef int (__stdcall *tLJM_LookupConstantName)(const char * Scope,
	double ConstantValue, char * ConstantName);

typedef void (__stdcall *tLJM_ErrorToString)(int ErrorCode, char * ErrorString);

typedef void (__stdcall *tLJM_LoadConstants)(void);

typedef int (__stdcall *tLJM_LoadConstantsFromFile)(const char * FileName);

typedef int (__stdcall *tLJM_LoadConstantsFromString)(const char * JsonString);

typedef int (__stdcall *tLJM_TCVoltsToTemp)(int TCType, double TCVolts, double CJTempK,
	double * pTCTempK);

typedef void (__stdcall *tLJM_FLOAT32ToByteArray)(const float * aFLOAT32, int RegisterOffset, int NumFLOAT32, unsigned char * aBytes);
typedef void (__stdcall *tLJM_ByteArrayToFLOAT32)(const unsigned char * aBytes, int RegisterOffset, int NumFLOAT32, float * aFLOAT32);

typedef void (__stdcall *tLJM_UINT16ToByteArray)(const unsigned short * aUINT16, int RegisterOffset, int NumUINT16, unsigned char * aBytes);
typedef void (__stdcall *tLJM_ByteArrayToUINT16)(const unsigned char * aBytes, int RegisterOffset, int NumUINT16, unsigned short * aUINT16);

typedef void (__stdcall *tLJM_UINT32ToByteArray)(const unsigned int * aUINT32, int RegisterOffset, int NumUINT32, unsigned char * aBytes);
typedef void (__stdcall *tLJM_ByteArrayToUINT32)(const unsigned char * aBytes, int RegisterOffset, int NumUINT32, unsigned int * aUINT32);

typedef void (__stdcall *tLJM_INT32ToByteArray)(const int * aINT32, int RegisterOffset, int NumINT32, unsigned char * aBytes);
typedef void (__stdcall *tLJM_ByteArrayToINT32)(const unsigned char * aBytes, int RegisterOffset, int NumINT32, int * aINT32);

typedef int (__stdcall *tLJM_NumberToIP)(unsigned int Number, char * IPv4String);

typedef int (__stdcall *tLJM_IPToNumber)(const char * IPv4String, unsigned int * Number);

typedef int (__stdcall *tLJM_NumberToMAC)(unsigned long long Number, char * MACString);

typedef int (__stdcall *tLJM_MACToNumber)(const char * MACString, unsigned long long * Number);

typedef long long (__stdcall *tLJM_GetHostTick)(void);

typedef void (__stdcall *tLJM_GetHostTick32Bit)(unsigned int * TickUpper, unsigned int * TickLower);

typedef int (__stdcall *tLJM_StartInterval)(int IntervalHandle, int Microseconds);

typedef int (__stdcall *tLJM_WaitForNextInterval)(int IntervalHandle, int * SkippedIntervals);

typedef int (__stdcall *tLJM_CleanInterval)(int IntervalHandle);

typedef int (__stdcall *tLJM_WriteLibraryConfigS)(const char * Parameter, double Value);

typedef int (__stdcall *tLJM_WriteLibraryConfigStringS)(const char * Parameter,
	const char * String);

typedef int (__stdcall *tLJM_ReadLibraryConfigS)(const char * Parameter, double * Value);

typedef int (__stdcall *tLJM_ReadLibraryConfigStringS)(const char * Parameter, char * String);

typedef int (__stdcall *tLJM_LoadConfigurationFile)(const char * FileName);

typedef int (__stdcall *tLJM_GetSpecificIPsInfo)(int * InfoHandle, const char ** Info);

typedef int (__stdcall *tLJM_GetDeepSearchInfo)(int * InfoHandle, const char ** Info);

typedef int (__stdcall *tLJM_Log)(int Level, const char * String);

typedef int (__stdcall *tLJM_ResetLog)(void);

typedef int (__stdcall *tLJM_RegisterDeviceReconnectCallback)(int Handle,
	LJM_DeviceReconnectCallback Callback);


//Define variables for the LJM functions.
tLJM_ListAll pLJM_ListAll;
tLJM_ListAllS pLJM_ListAllS;
tLJM_ListAllExtended pLJM_ListAllExtended;
tLJM_OpenS pLJM_OpenS;
tLJM_Open pLJM_Open;
tLJM_GetHandleInfo pLJM_GetHandleInfo;
tLJM_Close pLJM_Close;
tLJM_CloseAll pLJM_CloseAll;
tLJM_CleanInfo pLJM_CleanInfo;
tLJM_eWriteAddress pLJM_eWriteAddress;
tLJM_eReadAddress pLJM_eReadAddress;
tLJM_eWriteName pLJM_eWriteName;
tLJM_eReadName pLJM_eReadName;
tLJM_eReadAddresses pLJM_eReadAddresses;
tLJM_eReadNames pLJM_eReadNames;
tLJM_eWriteAddresses pLJM_eWriteAddresses;
tLJM_eWriteNames pLJM_eWriteNames;
tLJM_eReadAddressArray pLJM_eReadAddressArray;
tLJM_eReadNameArray pLJM_eReadNameArray;
tLJM_eWriteAddressArray pLJM_eWriteAddressArray;
tLJM_eWriteNameArray pLJM_eWriteNameArray;
tLJM_eReadAddressByteArray pLJM_eReadAddressByteArray;
tLJM_eReadNameByteArray pLJM_eReadNameByteArray;
tLJM_eWriteAddressByteArray pLJM_eWriteAddressByteArray;
tLJM_eWriteNameByteArray pLJM_eWriteNameByteArray;
tLJM_eAddresses pLJM_eAddresses;
tLJM_eReadNameString pLJM_eReadNameString;
tLJM_eReadAddressString pLJM_eReadAddressString;
tLJM_eWriteNameString pLJM_eWriteNameString;
tLJM_eWriteAddressString pLJM_eWriteAddressString;
tLJM_eStreamStart pLJM_eStreamStart;
tLJM_eStreamRead pLJM_eStreamRead;
tLJM_SetStreamCallback pLJM_SetStreamCallback;
tLJM_eStreamStop pLJM_eStreamStop;
tLJM_StreamBurst pLJM_StreamBurst;
tLJM_GetStreamTCPReceiveBufferStatus pLJM_GetStreamTCPReceiveBufferStatus;
tLJM_InitializeAperiodicStreamOut pLJM_InitializeAperiodicStreamOut;
tLJM_WriteAperiodicStreamOut pLJM_WriteAperiodicStreamOut;
tLJM_PeriodicStreamOut pLJM_PeriodicStreamOut;
tLJM_WriteRaw pLJM_WriteRaw;
tLJM_ReadRaw pLJM_ReadRaw;
tLJM_AddressesToMBFB pLJM_AddressesToMBFB;
tLJM_MBFBComm pLJM_MBFBComm;
tLJM_UpdateValues pLJM_UpdateValues;
tLJM_NamesToAddresses pLJM_NamesToAddresses;
tLJM_NameToAddress pLJM_NameToAddress;
tLJM_AddressesToTypes pLJM_AddressesToTypes;
tLJM_AddressToType pLJM_AddressToType;
tLJM_LookupConstantValue pLJM_LookupConstantValue;
tLJM_LookupConstantName pLJM_LookupConstantName;
tLJM_ErrorToString pLJM_ErrorToString;
tLJM_LoadConstants pLJM_LoadConstants;
tLJM_LoadConstantsFromFile pLJM_LoadConstantsFromFile;
tLJM_LoadConstantsFromString pLJM_LoadConstantsFromString;
tLJM_TCVoltsToTemp pLJM_TCVoltsToTemp;
tLJM_FLOAT32ToByteArray pLJM_FLOAT32ToByteArray;
tLJM_ByteArrayToFLOAT32 pLJM_ByteArrayToFLOAT32;
tLJM_UINT16ToByteArray pLJM_UINT16ToByteArray;
tLJM_ByteArrayToUINT16 pLJM_ByteArrayToUINT16;
tLJM_UINT32ToByteArray pLJM_UINT32ToByteArray;
tLJM_ByteArrayToUINT32 pLJM_ByteArrayToUINT32;
tLJM_INT32ToByteArray pLJM_INT32ToByteArray;
tLJM_ByteArrayToINT32 pLJM_ByteArrayToINT32;
tLJM_NumberToIP pLJM_NumberToIP;
tLJM_IPToNumber pLJM_IPToNumber;
tLJM_NumberToMAC pLJM_NumberToMAC;
tLJM_MACToNumber pLJM_MACToNumber;
tLJM_GetHostTick pLJM_GetHostTick;
tLJM_GetHostTick32Bit pLJM_GetHostTick32Bit;
tLJM_StartInterval pLJM_StartInterval;
tLJM_WaitForNextInterval pLJM_WaitForNextInterval;
tLJM_CleanInterval pLJM_CleanInterval;
tLJM_WriteLibraryConfigS pLJM_WriteLibraryConfigS;
tLJM_WriteLibraryConfigStringS pLJM_WriteLibraryConfigStringS;
tLJM_ReadLibraryConfigS pLJM_ReadLibraryConfigS;
tLJM_ReadLibraryConfigStringS pLJM_ReadLibraryConfigStringS;
tLJM_LoadConfigurationFile pLJM_LoadConfigurationFile;
tLJM_GetSpecificIPsInfo pLJM_GetSpecificIPsInfo;
tLJM_GetDeepSearchInfo pLJM_GetDeepSearchInfo;
tLJM_Log pLJM_Log;
tLJM_ResetLog pLJM_ResetLog;
tLJM_RegisterDeviceReconnectCallback pLJM_RegisterDeviceReconnectCallback;

#ifdef _WIN32
	//Define a variable to hold a handle to the loaded DLL.
	HINSTANCE ljm_library;

	//Start of the LoadLabJackM function.
	//This is the function used to dynamically load the DLL.
	void LoadLabJackM(void)
	{
		//Now try and load the DLL
		ljm_library = LoadLibrary("LabJackM.dll");
		if(ljm_library != NULL)
		{
			//If successfully loaded, get the address of the functions.
			pLJM_ListAll = (tLJM_ListAll) GetProcAddress(ljm_library, "LJM_ListAll");
			pLJM_ListAllS = (tLJM_ListAllS) GetProcAddress(ljm_library, "LJM_ListAllS");
			pLJM_ListAllExtended = (tLJM_ListAllExtended) GetProcAddress(ljm_library, "LJM_ListAllExtended");
			pLJM_OpenS = (tLJM_OpenS) GetProcAddress(ljm_library, "LJM_OpenS");
			pLJM_Open = (tLJM_Open) GetProcAddress(ljm_library, "LJM_Open");
			pLJM_GetHandleInfo = (tLJM_GetHandleInfo) GetProcAddress(ljm_library, "LJM_GetHandleInfo");
			pLJM_Close = (tLJM_Close) GetProcAddress(ljm_library, "LJM_Close");
			pLJM_CloseAll = (tLJM_CloseAll) GetProcAddress(ljm_library, "LJM_CloseAll");
			pLJM_CleanInfo = (tLJM_CleanInfo) GetProcAddress(ljm_library, "LJM_CleanInfo");
			pLJM_eWriteAddress = (tLJM_eWriteAddress) GetProcAddress(ljm_library, "LJM_eWriteAddress");
			pLJM_eReadAddress = (tLJM_eReadAddress) GetProcAddress(ljm_library, "LJM_eReadAddress");
			pLJM_eWriteName = (tLJM_eWriteName) GetProcAddress(ljm_library, "LJM_eWriteName");
			pLJM_eReadName = (tLJM_eReadName) GetProcAddress(ljm_library, "LJM_eReadName");
			pLJM_eReadAddresses = (tLJM_eReadAddresses) GetProcAddress(ljm_library, "LJM_eReadAddresses");
			pLJM_eReadNames = (tLJM_eReadNames) GetProcAddress(ljm_library, "LJM_eReadNames");
			pLJM_eWriteAddresses = (tLJM_eWriteAddresses) GetProcAddress(ljm_library, "LJM_eWriteAddresses");
			pLJM_eWriteNames = (tLJM_eWriteNames) GetProcAddress(ljm_library, "LJM_eWriteNames");
			pLJM_eReadAddressArray = (tLJM_eReadAddressArray) GetProcAddress(ljm_library, "LJM_eReadAddressArray");
			pLJM_eReadNameArray = (tLJM_eReadNameArray) GetProcAddress(ljm_library, "LJM_eReadNameArray");
			pLJM_eWriteAddressArray = (tLJM_eWriteAddressArray) GetProcAddress(ljm_library, "LJM_eWriteAddressArray");
			pLJM_eWriteNameArray = (tLJM_eWriteNameArray) GetProcAddress(ljm_library, "LJM_eWriteNameArray");
			pLJM_eReadAddressByteArray = (tLJM_eReadAddressByteArray) GetProcAddress(ljm_library, "LJM_eReadAddressByteArray");
			pLJM_eReadNameByteArray = (tLJM_eReadNameByteArray) GetProcAddress(ljm_library, "LJM_eReadNameByteArray");
			pLJM_eWriteAddressByteArray = (tLJM_eWriteAddressByteArray) GetProcAddress(ljm_library, "LJM_eWriteAddressByteArray");
			pLJM_eWriteNameByteArray = (tLJM_eWriteNameByteArray) GetProcAddress(ljm_library, "LJM_eWriteNameByteArray");
			pLJM_eAddresses = (tLJM_eAddresses) GetProcAddress(ljm_library, "LJM_eAddresses");
			pLJM_eReadNameString = (tLJM_eReadNameString) GetProcAddress(ljm_library, "LJM_eReadNameString");
			pLJM_eReadAddressString = (tLJM_eReadAddressString) GetProcAddress(ljm_library, "LJM_eReadAddressString");
			pLJM_eWriteNameString = (tLJM_eWriteNameString) GetProcAddress(ljm_library, "LJM_eWriteNameString");
			pLJM_eWriteAddressString = (tLJM_eWriteAddressString) GetProcAddress(ljm_library, "LJM_eWriteAddressString");
			pLJM_eStreamStart = (tLJM_eStreamStart) GetProcAddress(ljm_library, "LJM_eStreamStart");
			pLJM_eStreamRead = (tLJM_eStreamRead) GetProcAddress(ljm_library, "LJM_eStreamRead");
			pLJM_SetStreamCallback = (tLJM_SetStreamCallback) GetProcAddress(ljm_library, "LJM_SetStreamCallback");
			pLJM_eStreamStop = (tLJM_eStreamStop) GetProcAddress(ljm_library, "LJM_eStreamStop");
			pLJM_StreamBurst = (tLJM_StreamBurst) GetProcAddress(ljm_library, "LJM_StreamBurst");
			pLJM_GetStreamTCPReceiveBufferStatus = (tLJM_GetStreamTCPReceiveBufferStatus) GetProcAddress(ljm_library, "LJM_GetStreamTCPReceiveBufferStatus");
			pLJM_InitializeAperiodicStreamOut = (tLJM_InitializeAperiodicStreamOut) GetProcAddress(ljm_library, "LJM_InitializeAperiodicStreamOut");
			pLJM_WriteAperiodicStreamOut = (tLJM_WriteAperiodicStreamOut) GetProcAddress(ljm_library, "LJM_WriteAperiodicStreamOut");
			pLJM_PeriodicStreamOut = (tLJM_PeriodicStreamOut) GetProcAddress(ljm_library, "LJM_PeriodicStreamOut");
			pLJM_WriteRaw = (tLJM_WriteRaw) GetProcAddress(ljm_library, "LJM_WriteRaw");
			pLJM_ReadRaw = (tLJM_ReadRaw) GetProcAddress(ljm_library, "LJM_ReadRaw");
			pLJM_AddressesToMBFB = (tLJM_AddressesToMBFB) GetProcAddress(ljm_library, "LJM_AddressesToMBFB");
			pLJM_MBFBComm = (tLJM_MBFBComm) GetProcAddress(ljm_library, "LJM_MBFBComm");
			pLJM_UpdateValues = (tLJM_UpdateValues) GetProcAddress(ljm_library, "LJM_UpdateValues");
			pLJM_NamesToAddresses = (tLJM_NamesToAddresses) GetProcAddress(ljm_library, "LJM_NamesToAddresses");
			pLJM_NameToAddress = (tLJM_NameToAddress) GetProcAddress(ljm_library, "LJM_NameToAddress");
			pLJM_AddressesToTypes = (tLJM_AddressesToTypes) GetProcAddress(ljm_library, "LJM_AddressesToTypes");
			pLJM_AddressToType = (tLJM_AddressToType) GetProcAddress(ljm_library, "LJM_AddressToType");
			pLJM_LookupConstantValue = (tLJM_LookupConstantValue) GetProcAddress(ljm_library, "LJM_LookupConstantValue");
			pLJM_LookupConstantName = (tLJM_LookupConstantName) GetProcAddress(ljm_library, "LJM_LookupConstantName");
			pLJM_ErrorToString = (tLJM_ErrorToString) GetProcAddress(ljm_library, "LJM_ErrorToString");
			pLJM_LoadConstants = (tLJM_LoadConstants) GetProcAddress(ljm_library, "LJM_LoadConstants");
			pLJM_LoadConstantsFromFile = (tLJM_LoadConstantsFromFile) GetProcAddress(ljm_library, "LJM_LoadConstantsFromFile");
			pLJM_LoadConstantsFromString = (tLJM_LoadConstantsFromString) GetProcAddress(ljm_library, "LJM_LoadConstantsFromString");
			pLJM_TCVoltsToTemp = (tLJM_TCVoltsToTemp) GetProcAddress(ljm_library, "LJM_TCVoltsToTemp");
			pLJM_FLOAT32ToByteArray = (tLJM_FLOAT32ToByteArray) GetProcAddress(ljm_library, "LJM_FLOAT32ToByteArray");
			pLJM_ByteArrayToFLOAT32 = (tLJM_ByteArrayToFLOAT32) GetProcAddress(ljm_library, "LJM_ByteArrayToFLOAT32");
			pLJM_UINT16ToByteArray = (tLJM_UINT16ToByteArray) GetProcAddress(ljm_library, "LJM_UINT16ToByteArray");
			pLJM_ByteArrayToUINT16 = (tLJM_ByteArrayToUINT16) GetProcAddress(ljm_library, "LJM_ByteArrayToUINT16");
			pLJM_UINT32ToByteArray = (tLJM_UINT32ToByteArray) GetProcAddress(ljm_library, "LJM_UINT32ToByteArray");
			pLJM_ByteArrayToUINT32 = (tLJM_ByteArrayToUINT32) GetProcAddress(ljm_library, "LJM_ByteArrayToUINT32");
			pLJM_INT32ToByteArray = (tLJM_INT32ToByteArray) GetProcAddress(ljm_library, "LJM_INT32ToByteArray");
			pLJM_ByteArrayToINT32 = (tLJM_ByteArrayToINT32) GetProcAddress(ljm_library, "LJM_ByteArrayToINT32");
			pLJM_NumberToIP = (tLJM_NumberToIP) GetProcAddress(ljm_library, "LJM_NumberToIP");
			pLJM_IPToNumber = (tLJM_IPToNumber) GetProcAddress(ljm_library, "LJM_IPToNumber");
			pLJM_NumberToMAC = (tLJM_NumberToMAC) GetProcAddress(ljm_library, "LJM_NumberToMAC");
			pLJM_MACToNumber = (tLJM_MACToNumber) GetProcAddress(ljm_library, "LJM_MACToNumber");
			pLJM_GetHostTick = (tLJM_GetHostTick) GetProcAddress(ljm_library, "LJM_GetHostTick");
			pLJM_GetHostTick32Bit = (tLJM_GetHostTick32Bit) GetProcAddress(ljm_library, "LJM_GetHostTick32Bit");
			pLJM_StartInterval = (tLJM_StartInterval) GetProcAddress(ljm_library, "LJM_StartInterval");
			pLJM_WaitForNextInterval = (tLJM_WaitForNextInterval) GetProcAddress(ljm_library, "LJM_WaitForNextInterval");
			pLJM_CleanInterval = (tLJM_CleanInterval) GetProcAddress(ljm_library, "LJM_CleanInterval");
			pLJM_WriteLibraryConfigS = (tLJM_WriteLibraryConfigS) GetProcAddress(ljm_library, "LJM_WriteLibraryConfigS");
			pLJM_WriteLibraryConfigStringS = (tLJM_WriteLibraryConfigStringS) GetProcAddress(ljm_library, "LJM_WriteLibraryConfigStringS");
			pLJM_ReadLibraryConfigS = (tLJM_ReadLibraryConfigS) GetProcAddress(ljm_library, "LJM_ReadLibraryConfigS");
			pLJM_ReadLibraryConfigStringS = (tLJM_ReadLibraryConfigStringS) GetProcAddress(ljm_library, "LJM_ReadLibraryConfigStringS");
			pLJM_LoadConfigurationFile = (tLJM_LoadConfigurationFile) GetProcAddress(ljm_library, "LJM_LoadConfigurationFile");
			pLJM_GetSpecificIPsInfo = (tLJM_GetSpecificIPsInfo) GetProcAddress(ljm_library, "LJM_GetSpecificIPsInfo");
			pLJM_GetDeepSearchInfo = (tLJM_GetDeepSearchInfo) GetProcAddress(ljm_library, "LJM_GetDeepSearchInfo");
			pLJM_Log = (tLJM_Log) GetProcAddress(ljm_library, "LJM_Log");
			pLJM_ResetLog = (tLJM_ResetLog) GetProcAddress(ljm_library, "LJM_ResetLog");
			pLJM_RegisterDeviceReconnectCallback = (tLJM_RegisterDeviceReconnectCallback) GetProcAddress(ljm_library, "LJM_RegisterDeviceReconnectCallback");
			
		}
		else
		{
			printf("\nFailed to load DLL\n");
			getchar();
			exit(0);
		}
		// tLJM_Open now holds a pointer to the LJM_Open function.  The compiler
		// automatically recognizes tLJM_Open as a pointer to a function and
		// calls the function with the parameters given.  If we created another
		// variable of type pLJM_Open and simply put "pNewVar = pLJM_Open",
		// then the compiler might not know to call the function.
	}
	//End of LoadLabJackM function.

	// Function to free the Library
	void CloseLabJackM(void)
	{
		FreeLibrary(ljm_library);
	}

#else
	//Define a variable to hold a handle to the loaded DLL.
	void *ljm_library;

	//Start of the LoadLabJackM function.
	//This is the function used to dynamically load the DLL.
	void LoadLabJackM(void)
	{
		// Now try and load the library
		// Note: for the below to work, you need to link against libdl.
		// 		 A typical, basic C example should look like:
		// 		 gcc -g -Wall foo.c -ldl -o foo
		#ifdef __APPLE__
			ljm_library = dlopen("libLabJackM.dylib", RTLD_LAZY);
		#else
			ljm_library = dlopen("libLabJackM.so", RTLD_LAZY);
		#endif

		if(ljm_library != NULL)
		{
			//If successfully loaded, get the address of the functions.
			pLJM_ListAll = (tLJM_ListAll) dlsym(ljm_library, "LJM_ListAll");
			pLJM_ListAllS = (tLJM_ListAllS) dlsym(ljm_library, "LJM_ListAllS");
			pLJM_ListAllExtended = (tLJM_ListAllExtended) dlsym(ljm_library, "LJM_ListAllExtended");
			pLJM_OpenS = (tLJM_OpenS) dlsym(ljm_library, "LJM_OpenS");
			pLJM_Open = (tLJM_Open) dlsym(ljm_library, "LJM_Open");
			pLJM_GetHandleInfo = (tLJM_GetHandleInfo) dlsym(ljm_library, "LJM_GetHandleInfo");
			pLJM_Close = (tLJM_Close) dlsym(ljm_library, "LJM_Close");
			pLJM_CloseAll = (tLJM_CloseAll) dlsym(ljm_library, "LJM_CloseAll");
			pLJM_CleanInfo = (tLJM_CleanInfo) dlsym(ljm_library, "LJM_CleanInfo");
			pLJM_eWriteAddress = (tLJM_eWriteAddress) dlsym(ljm_library, "LJM_eWriteAddress");
			pLJM_eReadAddress = (tLJM_eReadAddress) dlsym(ljm_library, "LJM_eReadAddress");
			pLJM_eWriteName = (tLJM_eWriteName) dlsym(ljm_library, "LJM_eWriteName");
			pLJM_eReadName = (tLJM_eReadName) dlsym(ljm_library, "LJM_eReadName");
			pLJM_eReadAddresses = (tLJM_eReadAddresses) dlsym(ljm_library, "LJM_eReadAddresses");
			pLJM_eReadNames = (tLJM_eReadNames) dlsym(ljm_library, "LJM_eReadNames");
			pLJM_eWriteAddresses = (tLJM_eWriteAddresses) dlsym(ljm_library, "LJM_eWriteAddresses");
			pLJM_eWriteNames = (tLJM_eWriteNames) dlsym(ljm_library, "LJM_eWriteNames");
			pLJM_eReadAddressArray = (tLJM_eReadAddressArray) dlsym(ljm_library, "LJM_eReadAddressArray");
			pLJM_eReadNameArray = (tLJM_eReadNameArray) dlsym(ljm_library, "LJM_eReadNameArray");
			pLJM_eWriteAddressArray = (tLJM_eWriteAddressArray) dlsym(ljm_library, "LJM_eWriteAddressArray");
			pLJM_eWriteNameArray = (tLJM_eWriteNameArray) dlsym(ljm_library, "LJM_eWriteNameArray");
			pLJM_eReadAddressByteArray = (tLJM_eReadAddressByteArray) dlsym(ljm_library, "LJM_eReadAddressByteArray");
			pLJM_eReadNameByteArray = (tLJM_eReadNameByteArray) dlsym(ljm_library, "LJM_eReadNameByteArray");
			pLJM_eWriteAddressByteArray = (tLJM_eWriteAddressByteArray) dlsym(ljm_library, "LJM_eWriteAddressByteArray");
			pLJM_eWriteNameByteArray = (tLJM_eWriteNameByteArray) dlsym(ljm_library, "LJM_eWriteNameByteArray");
			pLJM_eAddresses = (tLJM_eAddresses) dlsym(ljm_library, "LJM_eAddresses");
			pLJM_eReadNameString = (tLJM_eReadNameString) dlsym(ljm_library, "LJM_eReadNameString");
			pLJM_eReadAddressString = (tLJM_eReadAddressString) dlsym(ljm_library, "LJM_eReadAddressString");
			pLJM_eWriteNameString = (tLJM_eWriteNameString) dlsym(ljm_library, "LJM_eWriteNameString");
			pLJM_eWriteAddressString = (tLJM_eWriteAddressString) dlsym(ljm_library, "LJM_eWriteAddressString");
			pLJM_eStreamStart = (tLJM_eStreamStart) dlsym(ljm_library, "LJM_eStreamStart");
			pLJM_eStreamRead = (tLJM_eStreamRead) dlsym(ljm_library, "LJM_eStreamRead");
			pLJM_SetStreamCallback = (tLJM_SetStreamCallback) dlsym(ljm_library, "LJM_SetStreamCallback");
			pLJM_eStreamStop = (tLJM_eStreamStop) dlsym(ljm_library, "LJM_eStreamStop");
			pLJM_StreamBurst = (tLJM_StreamBurst) dlsym(ljm_library, "LJM_StreamBurst");
			pLJM_GetStreamTCPReceiveBufferStatus = (tLJM_GetStreamTCPReceiveBufferStatus) dlsym(ljm_library, "LJM_GetStreamTCPReceiveBufferStatus");
			pLJM_InitializeAperiodicStreamOut = (tLJM_InitializeAperiodicStreamOut) dlsym(ljm_library, "LJM_InitializeAperiodicStreamOut");
			pLJM_WriteAperiodicStreamOut = (tLJM_WriteAperiodicStreamOut) dlsym(ljm_library, "LJM_WriteAperiodicStreamOut");
			pLJM_PeriodicStreamOut = (tLJM_PeriodicStreamOut) dlsym(ljm_library, "LJM_PeriodicStreamOut");
			pLJM_WriteRaw = (tLJM_WriteRaw) dlsym(ljm_library, "LJM_WriteRaw");
			pLJM_ReadRaw = (tLJM_ReadRaw) dlsym(ljm_library, "LJM_ReadRaw");
			pLJM_AddressesToMBFB = (tLJM_AddressesToMBFB) dlsym(ljm_library, "LJM_AddressesToMBFB");
			pLJM_MBFBComm = (tLJM_MBFBComm) dlsym(ljm_library, "LJM_MBFBComm");
			pLJM_UpdateValues = (tLJM_UpdateValues) dlsym(ljm_library, "LJM_UpdateValues");
			pLJM_NamesToAddresses = (tLJM_NamesToAddresses) dlsym(ljm_library, "LJM_NamesToAddresses");
			pLJM_NameToAddress = (tLJM_NameToAddress) dlsym(ljm_library, "LJM_NameToAddress");
			pLJM_AddressesToTypes = (tLJM_AddressesToTypes) dlsym(ljm_library, "LJM_AddressesToTypes");
			pLJM_AddressToType = (tLJM_AddressToType) dlsym(ljm_library, "LJM_AddressToType");
			pLJM_LookupConstantValue = (tLJM_LookupConstantValue) dlsym(ljm_library, "LJM_LookupConstantValue");
			pLJM_LookupConstantName = (tLJM_LookupConstantName) dlsym(ljm_library, "LJM_LookupConstantName");
			pLJM_ErrorToString = (tLJM_ErrorToString) dlsym(ljm_library, "LJM_ErrorToString");
			pLJM_LoadConstants = (tLJM_LoadConstants) dlsym(ljm_library, "LJM_LoadConstants");
			pLJM_LoadConstantsFromFile = (tLJM_LoadConstantsFromFile) dlsym(ljm_library, "LJM_LoadConstantsFromFile");
			pLJM_LoadConstantsFromString = (tLJM_LoadConstantsFromString) dlsym(ljm_library, "LJM_LoadConstantsFromString");
			pLJM_TCVoltsToTemp = (tLJM_TCVoltsToTemp) dlsym(ljm_library, "LJM_TCVoltsToTemp");
			pLJM_FLOAT32ToByteArray = (tLJM_FLOAT32ToByteArray) dlsym(ljm_library, "LJM_FLOAT32ToByteArray");
			pLJM_ByteArrayToFLOAT32 = (tLJM_ByteArrayToFLOAT32) dlsym(ljm_library, "LJM_ByteArrayToFLOAT32");
			pLJM_UINT16ToByteArray = (tLJM_UINT16ToByteArray) dlsym(ljm_library, "LJM_UINT16ToByteArray");
			pLJM_ByteArrayToUINT16 = (tLJM_ByteArrayToUINT16) dlsym(ljm_library, "LJM_ByteArrayToUINT16");
			pLJM_UINT32ToByteArray = (tLJM_UINT32ToByteArray) dlsym(ljm_library, "LJM_UINT32ToByteArray");
			pLJM_ByteArrayToUINT32 = (tLJM_ByteArrayToUINT32) dlsym(ljm_library, "LJM_ByteArrayToUINT32");
			pLJM_INT32ToByteArray = (tLJM_INT32ToByteArray) dlsym(ljm_library, "LJM_INT32ToByteArray");
			pLJM_ByteArrayToINT32 = (tLJM_ByteArrayToINT32) dlsym(ljm_library, "LJM_ByteArrayToINT32");
			pLJM_NumberToIP = (tLJM_NumberToIP) dlsym(ljm_library, "LJM_NumberToIP");
			pLJM_IPToNumber = (tLJM_IPToNumber) dlsym(ljm_library, "LJM_IPToNumber");
			pLJM_NumberToMAC = (tLJM_NumberToMAC) dlsym(ljm_library, "LJM_NumberToMAC");
			pLJM_MACToNumber = (tLJM_MACToNumber) dlsym(ljm_library, "LJM_MACToNumber");
			pLJM_GetHostTick = (tLJM_GetHostTick) dlsym(ljm_library, "LJM_GetHostTick");
			pLJM_GetHostTick32Bit = (tLJM_GetHostTick32Bit) dlsym(ljm_library, "LJM_GetHostTick32Bit");
			pLJM_StartInterval = (tLJM_StartInterval) dlsym(ljm_library, "LJM_StartInterval");
			pLJM_WaitForNextInterval = (tLJM_WaitForNextInterval) dlsym(ljm_library, "LJM_WaitForNextInterval");
			pLJM_CleanInterval = (tLJM_CleanInterval) dlsym(ljm_library, "LJM_CleanInterval");
			pLJM_WriteLibraryConfigS = (tLJM_WriteLibraryConfigS) dlsym(ljm_library, "LJM_WriteLibraryConfigS");
			pLJM_WriteLibraryConfigStringS = (tLJM_WriteLibraryConfigStringS) dlsym(ljm_library, "LJM_WriteLibraryConfigStringS");
			pLJM_ReadLibraryConfigS = (tLJM_ReadLibraryConfigS) dlsym(ljm_library, "LJM_ReadLibraryConfigS");
			pLJM_ReadLibraryConfigStringS = (tLJM_ReadLibraryConfigStringS) dlsym(ljm_library, "LJM_ReadLibraryConfigStringS");
			pLJM_LoadConfigurationFile = (tLJM_LoadConfigurationFile) dlsym(ljm_library, "LJM_LoadConfigurationFile");
			pLJM_GetSpecificIPsInfo = (tLJM_GetSpecificIPsInfo) dlsym(ljm_library, "LJM_GetSpecificIPsInfo");
			pLJM_GetDeepSearchInfo = (tLJM_GetDeepSearchInfo) dlsym(ljm_library, "LJM_GetDeepSearchInfo");
			pLJM_Log = (tLJM_Log) dlsym(ljm_library, "LJM_Log");
			pLJM_ResetLog = (tLJM_ResetLog) dlsym(ljm_library, "LJM_ResetLog");
			pLJM_RegisterDeviceReconnectCallback = (tLJM_RegisterDeviceReconnectCallback) dlsym(ljm_library, "LJM_RegisterDeviceReconnectCallback");
			
		}
		else
		{
			printf("\nFailed to load library\n");
			getchar();
			exit(0);
		}
		// tLJM_Open now holds a pointer to the LJM_Open function.  The compiler
		// automatically recognizes tLJM_Open as a pointer to a function and
		// calls the function with the parameters given.  If we created another
		// variable of type pLJM_Open and simply put "pNewVar = pLJM_Open",
		// then the compiler might not know to call the function.
	}
	//End of LoadLabJackM function.

	// Function to free the Library
	int CloseLabJackM(void)
	{
		return dlclose(ljm_library);
	}

#endif // #ifdef _WIN32

#ifdef __cplusplus
}
#endif

#endif // #define LABJACKM_DYNAMIC_HEADER
